<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Invoice;
use App\Models\InvoiceItem;
use Prgayman\Zatca\Facades\Zatca;
use Prgayman\Zatca\Utilis\QrCodeOptions;

class InvoiceController extends Controller
{
    public function index(Request $request)
    {
        // إنشاء استعلام الفواتير من قاعدة البيانات
        $query = Invoice::query();

        // إجمالي الفواتير (قبل التصفية)
        $totalCount = Invoice::count();

        // قراءة الفلاتر من الطلب
        $number = trim((string) $request->query('number', ''));
        $status = (string) $request->query('status', '');
        $dateFrom = (string) $request->query('date_from', '');
        $dateTo = (string) $request->query('date_to', '');
        $totalMin = (string) $request->query('total_min', '');
        $totalMax = (string) $request->query('total_max', '');

        if ($number !== '') {
            $query->where('number', 'like', '%' . $number . '%');
        }
        if ($status !== '') {
            $query->where('status', $status);
        }
        if ($dateFrom !== '') {
            $query->where('date', '>=', $dateFrom);
        }
        if ($dateTo !== '') {
            $query->where('date', '<=', $dateTo);
        }
        if ($totalMin !== '') {
            $query->where('total', '>=', (float) $totalMin);
        }
        if ($totalMax !== '') {
            $query->where('total', '<=', (float) $totalMax);
        }

        // ترتيب حسب العمود المطلوب
        $sortBy = (string) $request->query('sort_by', '');
        $sortDir = strtolower((string) $request->query('sort_dir', 'asc')) === 'desc' ? 'desc' : 'asc';
        $sortable = [
            'number' => 'number',
            'customer' => 'customer_name',
            'date' => 'date',
            'total' => 'total',
            'status' => 'status',
        ];
        if (array_key_exists($sortBy, $sortable)) {
            $query->orderBy($sortable[$sortBy], $sortDir);
        }

        // جلب النتائج وتحويلها لتوافق الواجهة الحالية
        $results = $query->get()->map(function (Invoice $inv) {
            return [
                'id' => $inv->id,
                'number' => $inv->number,
                'customer' => $inv->customer_name,
                'date' => $inv->date ? $inv->date->toDateString() : '',
                'total' => (float) $inv->total,
                'status' => $inv->status,
            ];
        });

        return view('invoices.index', [
            'invoices' => $results,
            'totalCount' => $totalCount,
        ]);
    }

    public function create(Request $request)
    {
        return view('invoices.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'customer_name' => ['required', 'string', 'max:255'],
            'customer_phone' => ['required', 'string', 'max:50'],
            'customer_address' => ['nullable', 'string', 'max:255'],
            'tax_number' => ['nullable', 'string', 'max:50'],
            'items' => ['required', 'array', 'min:1'],
            'items.*.description' => ['required', 'string', 'max:255'],
            'items.*.quantity' => ['required', 'numeric', 'min:1'],
            'items.*.unit_price' => ['required', 'numeric', 'min:0'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);

        DB::transaction(function () use ($validated) {
            $items = $validated['items'];
            $subtotal = 0;

            foreach ($items as $it) {
                $qty = (int) $it['quantity'];
                $price = (float) $it['unit_price'];
                $subtotal += $qty * $price;
            }

            $invoice = new Invoice();
            $invoice->number = 'INV-' . date('Ymd') . '-' . substr(uniqid(), -6);
            $invoice->customer_name = $validated['customer_name'];
            $invoice->customer_phone = $validated['customer_phone'];
            $invoice->customer_address = $validated['customer_address'] ?? null;
            $invoice->tax_number = $validated['tax_number'] ?? null;
            $invoice->date = now()->toDateString();
            $invoice->status = 'غير مسدد';
            $invoice->notes = $validated['notes'] ?? null;
            $invoice->subtotal = $subtotal;
-            $invoice->tax = 0;
-            $invoice->total = $subtotal;
+            // تطبيق ضريبة القيمة المضافة 15%
+            $invoice->tax = $subtotal * 0.15;
+            $invoice->total = $invoice->subtotal + $invoice->tax;
            $invoice->save();

            foreach ($items as $it) {
                $qty = (int) $it['quantity'];
                $price = (float) $it['unit_price'];
                $lineTotal = $qty * $price;

                $invoice->items()->create([
                    'description' => $it['description'],
                    'quantity' => $qty,
                    'unit_price' => $price,
                    'line_total' => $lineTotal,
                ]);
            }
        });

        return redirect()->route('invoices.index')->with('success', 'تم إنشاء الفاتورة بنجاح.');
    }

    public function show(Invoice $invoice)
    {
        $invoice->load('items');
        
        // تأكد من حساب الإجماليات والضريبة بناءً على البنود (لعلاج الفواتير القديمة)
        $calculatedSubtotal = (float) $invoice->items->sum('line_total');
        if ($calculatedSubtotal > 0 && ((float)$invoice->tax === 0.0 || (float)$invoice->subtotal !== $calculatedSubtotal)) {
            $invoice->subtotal = $calculatedSubtotal;
            $invoice->tax = round($calculatedSubtotal * 0.15, 2);
            $invoice->total = round($invoice->subtotal + $invoice->tax, 2);
            $invoice->save();
        }

        $setting = \App\Models\Setting::query()->first();
    
        $qrCodeSvg = null;
        if (!empty($setting?->company_name) && !empty($setting?->company_tax_number)) {
            $timestamp = ($invoice->created_at ?? now())->setTimezone('UTC')->format('Y-m-d\\TH:i:s\\Z');
            $totalWithVat = number_format((float)($invoice->total ?? 0), 2, '.', '');
            $vatTotal = number_format((float)($invoice->tax ?? 0), 2, '.', '');
            try {
                $options = new QrCodeOptions();
                $options->format('svg');
                $options->size(140);
                $options->margin(0);
                $qrCodeSvg = Zatca::sellerName($setting->company_name)
                    ->vatRegistrationNumber($setting->company_tax_number)
                    ->timestamp($timestamp)
                    ->totalWithVat($totalWithVat)
                    ->vatTotal($vatTotal)
                    ->toQrCode($options);
            } catch (\Throwable $e) {
                $qrCodeSvg = null;
            }
        }
    
        return view('invoices.show', [
            'invoice' => $invoice,
            'setting' => $setting,
            'qrCodeSvg' => $qrCodeSvg,
        ]);
    }

    public function destroy(Invoice $invoice)
    {
        DB::transaction(function () use ($invoice) {
            // حذف العناصر المرتبطة أولًا لضمان الاتساق
            $invoice->items()->delete();
            $invoice->delete();
        });

        return redirect()->route('invoices.index')->with('success', 'تم حذف الفاتورة بنجاح.');
    }

    public function edit(\App\Models\Invoice $invoice)
    {
        $invoice->load('items');
        return view('invoices.edit', ['invoice' => $invoice]);
    }

    public function update(\Illuminate\Http\Request $request, \App\Models\Invoice $invoice)
    {
        $validated = $request->validate([
            'customer_name' => ['required', 'string', 'max:255'],
            'customer_phone' => ['required', 'string', 'max:50'],
            'customer_address' => ['nullable', 'string', 'max:255'],
            'tax_number' => ['nullable', 'string', 'max:50'],
            'date' => ['nullable', 'date'],
            'status' => ['nullable', 'in:غير مسدد,مسدد,معلق'],
            'items' => ['required', 'array', 'min:1'],
            'items.*.description' => ['required', 'string', 'max:255'],
            'items.*.quantity' => ['required', 'numeric', 'min:1'],
            'items.*.unit_price' => ['required', 'numeric', 'min:0'],
            'notes' => ['nullable', 'string', 'max:1000'],
        ]);
    
        \Illuminate\Support\Facades\DB::transaction(function () use ($invoice, $validated) {
            $items = $validated['items'];
            $subtotal = 0.0;
    
            foreach ($items as $it) {
                $qty = (int) $it['quantity'];
                $price = (float) $it['unit_price'];
                $subtotal += $qty * $price;
            }
    
            $invoice->customer_name = $validated['customer_name'];
            $invoice->customer_phone = $validated['customer_phone'];
            $invoice->customer_address = $validated['customer_address'] ?? null;
            $invoice->tax_number = $validated['tax_number'] ?? null;
            if (isset($validated['date'])) {
                $invoice->date = $validated['date'];
            }
            if (isset($validated['status'])) {
                $invoice->status = $validated['status'];
            }
            $invoice->notes = $validated['notes'] ?? null;
            $invoice->subtotal = $subtotal;
            $invoice->tax = round($subtotal * 0.15, 2);
            $invoice->total = round($invoice->subtotal + $invoice->tax, 2);
            $invoice->save();
    
            // استبدال البنود: حذف القديمة وإنشاء المرسلة
            $invoice->items()->delete();
    
            foreach ($items as $it) {
                $qty = (int) $it['quantity'];
                $price = (float) $it['unit_price'];
                $lineTotal = $qty * $price;
    
                $invoice->items()->create([
                    'description' => $it['description'],
                    'quantity' => $qty,
                    'unit_price' => $price,
                    'line_total' => $lineTotal,
                ]);
            }
        });
    
        return redirect()->route('invoices.show', $invoice)->with('success', 'تم تحديث الفاتورة بنجاح.');
    }

    public function toggleStatus(Invoice $invoice)
    {
        $invoice->status = $invoice->status === 'مسدد' ? 'غير مسدد' : 'مسدد';
        $invoice->save();

        return redirect()->route('invoices.show', $invoice)->with('success', 'تم تحديث حالة الفاتورة.');
    }
}